"""
Basic Python Implementation example for Anybus-module Modbus-TCP
Data Type: Float Format A

Version: 202507-2

This document provides an example of how to implement the Modbus-TCP protocol in Python using the pyModbusTCP open-source library.
    Hardware Setup:
        Power Supply: A power supply compatible with the INT-MOD-ANY pluggable interface module.
        Interface Module: INT-MOD-ANY
        ModbusTCP Module: Anybus CompactCom M40 ModbusTCP (plugged into the INT-MOD-ANY module)
    Documentation:
        The implementation is based on the Fieldbus Implementation Guide, which is available on the website.
        (https://www.delta-elektronika.nl/anybus-module)
    Purpose:
        This example demonstrates how to approach a Modbus-TCP implementation using the guide and the mentioned hardware. It is intended for educational or development purposes.
    Important Notes:
        No error handling is included in this code.
        Users should be cautious and understand the risks of using the code without proper validation and safety checks.

Disclaimer
This code is provided "as is" without any guarantees or warranties. Delta Elektronika B.V.
is not responsible for any damages, losses, or issues arising from its use,
implementation, or modification.

"""

# Required Libraries
from pyModbusTCP.client import ModbusClient
import struct
import time # Optional

# --------------------------------------------------------------------------- #
# Numeric Literals and Data Types in Python (for Modbus use)
# --------------------------------------------------------------------------- #
# Python supports different numeric formats for register addresses and values:
#
# 1. Decimal (base 10):      Just write the number directly.
#      Example:  offset = 1024
#
# 2. Hexadecimal (base 16):  Prefix with '0x'. Common for Modbus register maps.
#      Example:  offset = 0x400  # same as 1024
#
# 3. Binary (base 2):        Prefix with '0b'. Used when working with bit flags.
#      Example:  control_word = 0b0000001000000001  # bits 0 and 9 set
#
# These are all treated as integers (type: int) in Python, so you can use them
# interchangeably in Modbus function calls like:
#   client.write_single_register(offset, value)
#
# --------------------------------------------------------------------------- #
# Floating-point values (REAL32) and Bitwise Representation
# --------------------------------------------------------------------------- #
# 4. 32-bit IEEE 754 Float:
#    When Modbus registers hold a float (REAL32), they are split across two 16-bit words.
#
#    To convert float → two registers:
#        low, high = struct.unpack(">HH", struct.pack(">f", float_value))
#
#    To convert two registers → float:
#        combined = (high << 16) | low
#        float_value = struct.unpack(">f", combined.to_bytes(4, 'big'))[0]
#
# This is used to write and read float values via:
#   client.write_multiple_registers(offset, [low, high])
#   client.read_holding_registers(offset, 2)


# --------------------------------------------------------------------------- #
# Methods used from pyModbusTCP for Read and Write
# --------------------------------------------------------------------------- #

# Modbus function READ_HOLDING_REGISTERS (0x03)
# read_holding_registers(reg_addr, reg_nb=1)
# Parameters
# reg_addr (int) – register address (0 to 65535)
# reg_nb (int) – number of registers to read (1 to 125)

# Returns
# registers list or None if fail

# Return type
# list of int or None


# Modbus function WRITE_MULTIPLE_REGISTERS (0x10)
# write_multiple_registers(regs_addr, regs_value)
#     Parameters
#     regs_addr (int) – registers address (0 to 65535)
#     regs_value (list) – registers values to write

#     Returns
#     True if write ok
#     Return type
#     bool

# --------------------------------------------------------------------------- #
# Register mapping from the implementation guide for Float Format
# --------------------------------------------------------------------------- #

register_map_Read = {
    # All readable param.
    'CVprg': (0x800, 2),  # (offset, word length)
    'CVmon': (0x802, 2),
    'CCprg': (0x804, 2),
    'CCmon': (0x806, 2),
    'CVlim': (0x808, 2),
    'CClim': (0x80A, 2),
    'Refresh_Counter': (0x80C, 1),
    'Status_Register_A': (0x80D, 1),
    'Status_Register_B': (0x80E, 1),
}

register_map_Write = {
    # All writable param.
    'CVprg': (0x000, 2),
    'CCprg': (0x002, 2),
    'RemCTRL': (0x004, 1),
}

# --------------------------------------------------------------------------- #
# General Helper functions
# --------------------------------------------------------------------------- #

def connect(address):
    """
    Open connection to the Anybus module
    """
    client = ModbusClient(address, port=502, unit_id=1, auto_open=True)
    client.open()
    return client


def disconnect(client):
    """
    Close the connection to the Anybus module
    """
    client.close()

# --------------------------------------------------------------------------- #
# Float Format A, Helper functions
# --------------------------------------------------------------------------- #

def parse_to_float(data):
    """
    Combine two 16bit words to 32bit and interpret as float
    """
    low, high = data
    combined = (high << 16) | low
    return struct.unpack('>f', struct.pack('>I', combined))[0]


def read_param(client, parameter):
    """
    Read parameter from Anybus module
    """
    # Unpack tuple from the register map
    offset, words = register_map_Read[parameter]
    data_rec = client.read_holding_registers(offset, words)  # FC 0x03
    # Parameter of single word length
    if words == 1:
        # All parameters of length 1 word are of type UINT16
        return int(data_rec[0])
    # Parameters of two words are of type REAL32, combine result to 32-bit and interpret as float
    elif words == 2:
        return round(parse_to_float(data_rec), 2)  # Rounded to two decimals


def float_to_16bit(value):
    """
    Unpack float value into 16bit values
    """
    high, low = struct.unpack(">HH", struct.pack(">f", float(value)))
    return low, high


def write_param(client, parameter, value):
    """
    Write parameter to Anybus module
    """
    # Unpack tuple from the register map
    offset, words = register_map_Write[parameter]
    # If single word param type, set param
    if words == 1:
        # Expected is a UINT16 value
        client.write_single_register(offset, int(value))
    # Parameters of two words are of type REAL32, combine result to 32-bit and interpret as float
    elif words == 2:
        low, high = float_to_16bit(float(value))
        client.write_multiple_registers(offset, [low, high])

# --------------------------------------------------------------------------- #
# Additional Helper Functions
# --------------------------------------------------------------------------- #

def decode_status_register_a(value):
    """
    Decode the binary value representing Status Register A
    """
    value = int(value)
    bits = {
        0: "CV", 1: "CC", 2: "CP", 3: "Vlim", 4: "Ilim", 5: "Plim",
        6: "DcfVolt", 7: "DcfCurr", 8: "OT", 9: "PSOL", 10: "ACF",
        11: "Interlock", 12: "RSD", 13: "Output", 14: "FrontpanelLock"
    }
    return {name: bool(value & (1 << bit)) for bit, name in bits.items()}


def decode_status_register_b(value):
    """
    Decode the binary value representing Status Register B
    """
    value = int(value)
    bits = {
        0: "RemCV", 1: "RemCC", 2: "RemCP", 3: "ProgRunning",
        4: "WaitForTrigger", 5: "Master", 6: "Slave", 7: "VoutOverload",
        8: "IoutOverload", 13: "SenseBreak", 14: "PROT", 15: "ProgOpenEndError"
    }
    return {name: bool(value & (1 << bit)) for bit, name in bits.items()}


def encode_remctrl(rsd, output, rem_cv, rem_cc):
    """
    Return the 16-bit word for the RemCTRL register (flags must be 0 or 1).
    """
    if not all(flag in (0, 1) for flag in (rsd, output, rem_cv, rem_cc)):
        raise ValueError("All flags must be 0 or 1")
    return rsd | (output << 1) | (rem_cv << 9) | (rem_cc << 10)


def update_remctrl(client, rsd=None, output=None, rem_cv=None, rem_cc=None):
    """
    Update the RemCTRL register without changing current settings
    """
    # Read and decode present states
    flags_a = decode_status_register_a(read_param(client, "Status_Register_A"))
    flags_b = decode_status_register_b(read_param(client, "Status_Register_B"))

    # Keep current value if caller did not supply one
    if rsd is None:
        rsd = int(flags_a["RSD"])
    if output is None:
        output = int(flags_a["Output"])
    if rem_cv is None:
        rem_cv = int(flags_b["RemCV"])
    if rem_cc is None:
        rem_cc = int(flags_b["RemCC"])

    # Write the new control word
    word = encode_remctrl(rsd, output, rem_cv, rem_cc)
    write_param(client, "RemCTRL", word)


def read_print_all(client):
    """
    Print all parameters in pretty print formatting
    """
    # Read param
    cv_lim = read_param(client, "CVlim")   # V
    cc_lim = read_param(client, "CClim")   # A
    cv_prg = read_param(client, "CVprg")   # V
    cc_prg = read_param(client, "CCprg")   # A
    cv_mon = read_param(client, "CVmon")   # V
    cc_mon = read_param(client, "CCmon")   # A
    refresh = read_param(client, "Refresh_Counter")
    raw_a = read_param(client, "Status_Register_A")
    raw_b = read_param(client, "Status_Register_B")

    # Print
    print("\n====================================")
    print(f"CVprg:                  {cv_prg:.2f} V")
    print(f"CCprg:                  {cc_prg:.2f} A\n")
    print(f"CVmon:                  {cv_mon:.2f} V")
    print(f"CCmon:                  {cc_mon:.2f} A\n")
    print(f"CVlim:                  {cv_lim:.2f} V")
    print(f"CClim:                  {cc_lim:.2f} A\n")
    print(f"Status_Register_A:      {raw_a:016b}")
    print(f"Status_Register_B:      {raw_b:016b}\n")
    print(f"Refresh Counter:        {refresh}")
    print("====================================\n")


def read_print_registers_a_b(client):
    """
    Pretty print of status registers A and B
    """
    # Read and decode
    decoded_a = decode_status_register_a(
        read_param(client, "Status_Register_A"))
    decoded_b = decode_status_register_b(
        read_param(client, "Status_Register_B"))

    # Print vertically
    print("\n====================================")
    print("Status Register A")
    for name, state in decoded_a.items():
        print(f"  {name:<18}: {'ON' if state else 'off'}")

    print("\nStatus Register B")
    for name, state in decoded_b.items():
        print(f"  {name:<18}: {'ON' if state else 'off'}")
    print("====================================\n")

# --------------------------------------------------------------------------- #
# Demonstration Examples
# --------------------------------------------------------------------------- #

# Steps before connecting:
# 1. Configure all hardware as described in the product manuals.
# 2. Obtain the IP-address of the Anybus module through the web interface of the power supply.
# 3. Set the INT-MOD-ANY data format to Float format in the web interface of the power supply.
# 4. Safety first, dont connect any hardware to the Output of the power supply. Its at own risk.

# The demonstration examples show how programming and monitoring can be achieved.
# Functions are basic, it is assumed that when the reader understands the basic operation,
# that it is easy to customize or to add error handling or to wrap the code into a class.

# Configuration, IP-address of the Anybus module plugged into the INT-MOD-ANY
IPADDRESS_ANYBUSMODULE = "10.1.1.26"

# Open connection with the Anybus module, should be closed later on.
Anybus_connection = connect(IPADDRESS_ANYBUSMODULE)

# Note: To read parameters from the PSU, the programming source of the power supply,
# must be set to the INT-MOD-ANY. This can be done in multiple ways.
# By selecting it in the PSU's web server or sending remCV, remCC, remCP commands.

# Note: Time-delays are added optionally.

# Example 1: Read a parameter of float type.
data = read_param(Anybus_connection, 'CCmon')
print(data)

# Example 2: Read a parameter of register type.
data = read_param(Anybus_connection, 'Status_Register_A')
print(data)

# Additional: Decode the previously read Status Register A
decoded_data = decode_status_register_a(data)
print(decoded_data)
print(decoded_data["CV"])  # extract single value.

# Same for status register B
data = read_param(Anybus_connection, 'Status_Register_B')
print(data)

decoded_data = decode_status_register_b(data)
print(decoded_data)

# Example 3: Write a parameter of float type.
write_param(Anybus_connection, 'CVprg', 0.0) # Replace zero with desired voltage [V]
time.sleep(0.1)

# Example 4: Write a parameter of register type.
write_param(Anybus_connection, 'RemCTRL', 2) # Sets OUTPUT to 1 and all others to 0.
time.sleep(0.1)

# Example 5: Write a parameter of register type, but set multiple settings at once.
RSD = 0
OUTPUT = 1
REM_CV = 1
REM_CC = 1
write_param(Anybus_connection, 'RemCTRL',
            encode_remctrl(RSD, OUTPUT, REM_CV, REM_CC))
time.sleep(0.1)

# Example 6: Write only one register type setting, update OUTPUT in RemCTRL,
#  without changing the others
update_remctrl(Anybus_connection, output=1)
time.sleep(0.1)

# Clear RSD and set REM_CC, leave others untouched
update_remctrl(Anybus_connection, rsd=0, rem_cc=1)
time.sleep(0.1)

# Additional Example: Print all readable parameters
read_print_all(Anybus_connection)
time.sleep(0.1)

# Additional Example: Read and print registers in human friendly format
read_print_registers_a_b(Anybus_connection)
time.sleep(0.1)

# Close the connection
disconnect(Anybus_connection)
